<?php
/**
 * register_response.php
 * - Validates POST
 * - Checks duplicate by staff_id
 * - Inserts into DB (using db.php -> $conn mysqli)
 * - Renders success/fail page
 * - On success, emails a styled HTML receipt using PHPMailer (SMTP)
 */

require_once __DIR__ . '/db.php';

/* -----------------------------
   Read & sanitize input
----------------------------- */
$fullname   = trim($_POST['fullname'] ?? '');
$staff_id   = trim($_POST['staff_id'] ?? '');
$department = trim($_POST['department'] ?? '');
$dept_other = trim($_POST['department_other'] ?? '');
$email      = trim($_POST['email'] ?? '');
$dietary    = trim($_POST['dietary'] ?? '');
$transport  = trim($_POST['transport'] ?? '');

/* -----------------------------
   Validate (all required)
----------------------------- */
$errors = [];
if ($fullname === '')   $errors[] = "Name is required.";
if ($staff_id === '')   $errors[] = "Staff ID is required.";
if ($department === '') $errors[] = "Department is required.";
if ($email === '')      $errors[] = "Email is required.";
if ($dietary === '')    $errors[] = "Dietary preference is required.";
if ($transport === '')  $errors[] = "Transport selection is required.";

// If "Others" selected, require free-text and replace $department
if ($department !== '' && preg_match('/^other(s)?/i', $department)) {
  if ($dept_other === '') {
    $errors[] = "Please specify your Department in the 'Others' field.";
  } else {
    $department = $dept_other;
  }
}

// Email format
if ($email !== '' && !filter_var($email, FILTER_VALIDATE_EMAIL)) {
  $errors[] = "Invalid email address.";
}

// Early exit on validation errors
if (!empty($errors)) {
  renderPage(false, $errors);
  exit;
}

/* -----------------------------
   Duplicate check (by Staff ID)
----------------------------- */
$dupStmt = $conn->prepare("SELECT 1 FROM registrations WHERE staff_id = ? LIMIT 1");
if ($dupStmt) {
  $dupStmt->bind_param("s", $staff_id);
  $dupStmt->execute();
  $dupStmt->store_result();
  if ($dupStmt->num_rows > 0) {
    $dupStmt->close();
    renderPage(false, ["A registration already exists for Staff ID <strong>" . h($staff_id) . "</strong>. If you need to update your details, please contact the organizers."]);
    exit;
  }
  $dupStmt->close();
}

/* -----------------------------
   Insert into DB
----------------------------- */
$stmt = $conn->prepare("INSERT INTO registrations (fullname, staff_id, department, email, dietary, transport) VALUES (?,?,?,?,?,?)");
if (!$stmt) {
  renderPage(false, ["Database error (prepare)."]);
  exit;
}
$stmt->bind_param("ssssss", $fullname, $staff_id, $department, $email, $dietary, $transport);

if ($stmt->execute()) {
  $payload = [
    "fullname"   => $fullname,
    "staff_id"   => $staff_id,
    "department" => $department,
    "email"      => $email,
    "dietary"    => $dietary,
    "transport"  => $transport,
  ];

  // Send HTML email (ignore if mail fails; user still sees success page)
  sendSuccessEmail($payload);

  renderPage(true, [], $payload);
} else {
  // If UNIQUE index is present, catch duplicate error gracefully
  if ($stmt->errno == 1062) {
    renderPage(false, ["A registration already exists for Staff ID <strong>" . h($staff_id) . "</strong>. If you need to update your details, please contact the organizers."]);
  } else {
    renderPage(false, ["Database error (execute)."]);
  }
}

$stmt->close();
$conn->close();
exit;

/* =============================
   VIEW RENDERER (HTML page)
============================= */
function renderPage(bool $success, array $errors = [], array $data = []) {
  $title = $success ? "Registration Successful" : "Submission Failed";

  if ($success) {
    $content = '
      <h1>Registration Successful</h1>
      <div class="success">
        <p><strong>Thank you for registering!</strong><br>
        You will receive your official invitation closer to the event date.</p>

        <p>If you&rsquo;d like to perform and boogie on stage on that fateful night, don&rsquo;t forget to register here:<br>
        <a href="https://singevent.com/e/HSBCDND2025/#performances" target="_blank" rel="noopener">https://singevent.com/e/HSBCDND2025/#performances</a></p>

        <div class="grid">
          <div><strong>Name</strong><span>' . h($data["fullname"] ?? "") . '</span></div>
          <div><strong>Staff ID</strong><span>' . h($data["staff_id"] ?? "") . '</span></div>
          <div><strong>Department</strong><span>' . h($data["department"] ?? "") . '</span></div>
          <div><strong>Email</strong><span>' . h($data["email"] ?? "") . '</span></div>
          <div><strong>Dietary</strong><span>' . h($data["dietary"] ?? "") . '</span></div>
          <div><strong>Transport</strong><span>' . h($data["transport"] ?? "") . '</span></div>
        </div>

        <p class="ticket">Ticket Valid for 1 Person</p>
      </div>';
  } else {
    $content = '
      <h1>Submission Failed</h1>
      <div class="errors"><ul>';
    foreach ($errors as $e) {
      $content .= '<li>' . $e . '</li>';
    }
    $content .= '</ul></div>
      <div class="actions">
        <a class="btn" href="register.php">Go Back</a>
      </div>';
  }

  echo '<!doctype html>
<html lang="en">
<head>
<meta charset="utf-8"/>
<meta name="viewport" content="width=device-width, initial-scale=1"/>
<title>' . h($title) . '</title>
<link href="https://cdnjs.cloudflare.com/ajax/libs/normalize/8.0.1/normalize.min.css" rel="stylesheet">
<style>
  @font-face {
    font-family: "BebasKai";
    src: url("fonts/BebasKai.ttf") format("truetype");
    font-weight: normal;
    font-style: normal;
  }
  :root{ --btn:#99191a; --btn-dark:#6e1213; }
  html,body{height:100%}
  body{
    margin:0; font-family:"BebasKai",sans-serif;
    background:#000 url("images/Registration-BGG.jpg") center/cover fixed no-repeat;
    display:grid; place-items:center; padding:20px;
  }
  .card{
    width:min(760px, 94vw);
    background: rgba(255,255,255,0.92);
    border-radius:14px;
    padding:20px 18px;
    box-shadow:0 18px 50px rgba(0,0,0,.35);
    color:#1c1c1c;
    text-align:center;
  }
  h1{
    margin:6px 0 14px; letter-spacing:.14em; text-transform:uppercase;
    color:' . ($success ? '#1c1c1c' : '#99191a') . ';
    font-size: clamp(24px, 5vw, 40px);
  }
  .success{padding:12px; border-radius:10px; background:rgba(255,255,255,.96); color:#1c1c1c;}
  .errors{padding:12px; border-radius:10px; background:#99191a; color:#fff; text-align:left;}
  .errors ul{margin:0; padding-left:18px;}
  .grid{
    display:grid; grid-template-columns: 1fr 1fr; gap:10px 16px;
    margin: 12px auto 6px; text-align:left; max-width:640px;
  }
  .grid div{
    display:flex; justify-content:space-between; gap:16px;
    padding:8px 10px; background:rgba(0,0,0,0.04); border-radius:8px;
  }
  .grid strong{letter-spacing:.08em;}
  .grid span{opacity:.9; word-break:break-word;}
  .ticket{
    margin-top:12px; font-size:clamp(14px, 3.8vw, 18px);
    letter-spacing:.12em; text-transform:uppercase; color:#99191a;
    background: rgba(255,255,255,.8); padding:6px 10px; border-radius:8px; display:inline-block;
  }
  .actions{margin-top:16px; display:flex; gap:10px; justify-content:center; flex-wrap:wrap;}
  .btn{
    display:inline-block; text-decoration:none; text-align:center; min-width:180px;
    background:linear-gradient(180deg, var(--btn), var(--btn-dark));
    color:#fff; padding:12px 18px; border-radius:10px;
    letter-spacing:.08em; font-size:1.05rem;
    box-shadow:0 10px 24px rgba(0,0,0,.35);
  }
  /* Mobile tweaks */
  @media (max-width: 560px){
    .grid{ grid-template-columns: 1fr; gap:8px; }
    .grid div{ padding:8px 10px; }
    body{ padding:14px; }
    .card{ padding:16px 14px; }
  }
</style>
</head>
<body>
  <div class="card">' . $content . '</div>
</body>
</html>';
}

/* =============================
   EMAIL SENDER (PHPMailer)
============================= */
function sendSuccessEmail(array $data){
  // Include PHPMailer (non-Composer) from your uploaded ZIP
  require_once __DIR__ . '/PHPMailer/src/PHPMailer.php';
  require_once __DIR__ . '/PHPMailer/src/SMTP.php';
  require_once __DIR__ . '/PHPMailer/src/Exception.php';

  $mail = new PHPMailer\PHPMailer\PHPMailer(true);
  try {
    /* ---------- SMTP SETTINGS: replace with your credentials ---------- */
    $mail->isSMTP();
    $mail->Host       = 'mail.singevent.com';   // <-- TODO
    $mail->SMTPAuth   = true;
    $mail->Username   = 'no-reply@singevent.com'; // <-- TODO
    $mail->Password   = 'jvlQqXirF{36U9~B';    // <-- TODO
    $mail->SMTPSecure = PHPMailer\PHPMailer\PHPMailer::ENCRYPTION_STARTTLS; // or ENCRYPTION_SMTPS
    $mail->Port       = 587; // 465 for SMTPS
    $mail->CharSet    = 'UTF-8';
    /* ------------------------------------------------------------------ */

    // From / To
    $mail->setFrom('no-reply@singevent.com', 'HSBC D&D 2025');
    $mail->addAddress($data['email'] ?? '', $data['fullname'] ?? '');

    // (Optional) send a copy to organizer
    // $mail->addBCC('organizer@singevent.com', 'Organizer');

    // Subject
    $mail->isHTML(true);
    $mail->Subject = 'HSBC D&D 2025  Registration Successful';

    // Basic email-safe styles (inline)
    $styleTable = 'style="width:100%;border-collapse:collapse;background:#f7f7f7;border-radius:6px;font-family:Arial,Helvetica,sans-serif;font-size:14px"';
    $styleTdKey = 'style="padding:10px 12px;border-bottom:1px solid #e8e8e8;font-weight:bold;width:35%;vertical-align:top"';
    $styleTdVal = 'style="padding:10px 12px;border-bottom:1px solid #e8e8e8;vertical-align:top"';

    // Body (keep simple for mail clients)
    $mail->Body = '
      <div style="max-width:640px;margin:auto;background:#fffbe8;padding:20px 18px;border-radius:10px;border:1px solid #ecd9a6">
        <h2 style="margin:0 0 12px 0;text-align:center;color:#99191a;text-transform:uppercase;letter-spacing:.1em;font-family:Arial,Helvetica,sans-serif">
          Registration Successful
        </h2>

        <p style="margin:0 0 8px 0;font-family:Arial,Helvetica,sans-serif;color:#222">
          <strong>Thank you for registering!</strong><br>
          You will receive your official invitation closer to the event date.
        </p>

        <p style="margin:0 0 14px 0;font-family:Arial,Helvetica,sans-serif;color:#222">
          If youd like to perform and boogie on stage on that fateful night, dont forget to register here:<br>
          <a href="https://singevent.com/e/HSBCDND2025/#performances" style="color:#0366d6">https://singevent.com/e/HSBCDND2025/#performances</a>
        </p>

        <table ' . $styleTable . '>
          <tr><td ' . $styleTdKey . '>Name</td><td ' . $styleTdVal . '>' . h($data["fullname"]) . '</td></tr>
          <tr><td ' . $styleTdKey . '>Staff ID</td><td ' . $styleTdVal . '>' . h($data["staff_id"]) . '</td></tr>
          <tr><td ' . $styleTdKey . '>Department</td><td ' . $styleTdVal . '>' . h($data["department"]) . '</td></tr>
          <tr><td ' . $styleTdKey . '>Email</td><td ' . $styleTdVal . '>' . h($data["email"]) . '</td></tr>
          <tr><td ' . $styleTdKey . '>Dietary</td><td ' . $styleTdVal . '>' . h($data["dietary"]) . '</td></tr>
          <tr><td ' . $styleTdKey . '>Transport</td><td ' . $styleTdVal . '>' . h($data["transport"]) . '</td></tr>
        </table>

        <p style="margin:14px 0 0 0;text-align:center;color:#99191a;font-weight:bold;letter-spacing:.1em;font-family:Arial,Helvetica,sans-serif;text-transform:uppercase">
          Ticket Valid for 1 Person
        </p>
      </div>';

    $mail->AltBody =
      "Registration Successful\n\n" .
      "Thank you for registering! You will receive your official invitation closer to the event date.\n\n" .
      "Performances: https://singevent.com/e/HSBCDND2025/#performances\n\n" .
      "Name: " . ($data['fullname'] ?? '') . "\n" .
      "Staff ID: " . ($data['staff_id'] ?? '') . "\n" .
      "Department: " . ($data['department'] ?? '') . "\n" .
      "Email: " . ($data['email'] ?? '') . "\n" .
      "Dietary: " . ($data['dietary'] ?? '') . "\n" .
      "Transport: " . ($data['transport'] ?? '') . "\n" .
      "Ticket Valid for 1 Person\n";

    $mail->send();
  } catch (Throwable $e) {
    // Dont block the flow if the email fails  just log it
    error_log('PHPMailer error: ' . ($e->getMessage() ?? 'unknown'));
  }
}

/* -----------------------------
   HTML escape helper
----------------------------- */
function h($s){ return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }
